/************************************************************************/
/*                                                                      */
/* Borland Enterprise Core Objects                                      */
/*                                                                      */
/* Copyright (c) 2003-2005 Borland Software Corporation                 */
/*                                                                      */
/************************************************************************/

using System;
using System.Collections;
using System.ComponentModel;
using System.Windows.Forms;
using System.Reflection;
using System.Drawing;
#if DEBUG
//using System.Diagnostics;
#endif
using Borland.Eco.Handles;
using Borland.Eco.ObjectRepresentation;
using Borland.Eco.UmlRt;
using Borland.Eco.Globalization;

namespace Borland.Eco.WinForm
{
	[ProvideProperty("EcoDragSource", typeof(Control))]
	[ProvideProperty("EcoDropTarget", typeof(Control))]
	[ToolboxBitmap(typeof(EcoDragDropExtender), "Borland.Eco.WinForm.EcoDragDropExtender.bmp")]
	[ToolboxItem(true)]
	[ToolboxItemFilter("System.Windows.Forms")]
	public class EcoDragDropExtender: System.ComponentModel.Component, System.ComponentModel.IExtenderProvider
	{
		///<summary>
		///Helps determine tha appropriate drag-drop effect based on e.AllowedEffect and key state.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="e"/> is null.</exception>
		public static DragDropEffects AppropriateEffect(System.Windows.Forms.DragEventArgs e)
		{
			if (e == null) throw new ArgumentNullException("e"); // Do not localize
			DragDropEffects Result = DragDropEffects.None;
			// Set the effect based upon the KeyState.
			if ((e.KeyState & (8 + 32)) == (8 + 32) &&
				(e.AllowedEffect & DragDropEffects.Link) == DragDropEffects.Link)
			{	// KeyState 8 + 32 = CTL + ALT
				// Link drag and drop effect.
				Result = DragDropEffects.Link;
			}
			else if ((e.KeyState & 32) == 32 &&
				(e.AllowedEffect & DragDropEffects.Link) == DragDropEffects.Link)
			{ // ALT KeyState for link.
				Result = DragDropEffects.Link;
			}
			else if ((e.KeyState & 4) == 4 &&
				(e.AllowedEffect & DragDropEffects.Move) == DragDropEffects.Move)
			{ // SHIFT KeyState for move.
				Result = DragDropEffects.Move;
			}
			else if ((e.KeyState & 8) == 8 &&
				(e.AllowedEffect & DragDropEffects.Copy) == DragDropEffects.Copy)
			{ // CTL KeyState for copy.
				Result = DragDropEffects.Copy;
			}
			else if ((e.AllowedEffect & DragDropEffects.Link) == DragDropEffects.Link)
			{ // By default, the drop action should be link, if allowed.
				Result = DragDropEffects.Link;
			}
			return Result;
		}
		private Hashtable dragSourceComponents = new Hashtable();
		private Hashtable dropTargetComponents = new Hashtable();

		#region IExtenderProvider implementation
		public virtual bool CanExtend(object extendee)
		{
			return ((extendee is ListBox) || (extendee is DataGrid)); // only hook to certain types
		}
		#endregion

		#region Provided properties
		#region EcoDragSource
		[LocalizableCategory(typeof(EcoDragDropExtender), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(EcoDragDropExtender), "sPropertyEcoDragSource")]				
		[DefaultValue(false)]
		public bool GetEcoDragSource(Control control)
		{
			return dragSourceComponents[control] == null ? false : (bool)dragSourceComponents[control];
		}
		///<summary>
		///This virtual method hooks up the extended control's event. You may want some other event hooked, or hook to something else.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		protected virtual void HookupSource(Control control)
		{
			if (control == null) throw new ArgumentNullException("control"); // Do not localize
			control.MouseDown += new MouseEventHandler(OnMouseDown);
			control.MouseMove += new MouseEventHandler(OnMouseMove);
			control.MouseUp += new MouseEventHandler(OnMouseUp);
		}
		///<summary>
		///This virtual method unhooks up the extended controls event.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		protected virtual void UnhookSource(Control control)
		{
			if (control == null) throw new ArgumentNullException("control"); // Do not localize
			control.MouseDown -= new MouseEventHandler(OnMouseDown);
			control.MouseMove -= new MouseEventHandler(OnMouseMove);
			control.MouseUp -= new MouseEventHandler(OnMouseUp);
		}
		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		public void SetEcoDragSource(Control control, bool value)
		{
			if (control == null) throw new ArgumentNullException("control"); // Do not localize
			if (!value)
			{
				dragSourceComponents.Remove(control);
				UnhookSource(control);
			}
			else
			{
				dragSourceComponents[control] = value;
				HookupSource(control);
			}
		}
		#endregion
		#region EcoDropTarget
		[LocalizableCategory(typeof(EcoDragDropExtender), "sCategoryEcoGui")]
		[LocalizableDescription(typeof(EcoDragDropExtender), "sPropertyEcoDropTarget")]				
		[DefaultValue(false)]
		public bool GetEcoDropTarget(Control control)
		{
			return dropTargetComponents[control] == null ? false : (bool)dropTargetComponents[control];
		}
		///<summary>
		///This virtual method hooks up the extended control's event. You may want some other event hooked, or hook to something else.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		protected virtual void HookupTarget(Control control)
		{
			if (control == null) throw new ArgumentNullException("control"); // Do not localize
			control.DragDrop += new DragEventHandler(OnDragDrop);
			control.DragOver += new DragEventHandler(OnDragOver);
		}
		///<summary>
		///This virtual method unhooks up the extended controls event.
		///</summary>
		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		protected virtual void UnhookTarget(Control control)
		{
			if (control == null) throw new ArgumentNullException("control"); // Do not localize
			control.DragDrop -= new DragEventHandler(OnDragDrop);
			control.DragOver -= new DragEventHandler(OnDragOver);
		}
		///<exception cref="ArgumentNullException">Thrown if <paramref name="control"/> is null.</exception>
		public void SetEcoDropTarget(Control control, bool value)
		{
			if (control == null) throw new ArgumentNullException("control"); // Do not localize
			if (!value)
			{
				dropTargetComponents.Remove(control);
				UnhookTarget(control);
			}
			else
			{
				dropTargetComponents[control] = value;
				control.AllowDrop = true;
				HookupTarget(control);
			}
		}

		#endregion
		#endregion
		#region Utility code
		private static bool ElementConformsToList(IElement draggedElement, IElementCollection elementCollection)
		{
			// Check presence of objects
			if ((draggedElement == null) || (elementCollection == null))
				return false;
			
			IEcoClassifier draggedclassifier = draggedElement.UmlType.EcoClassifier;
			#if DEBUG
			//Debug.WriteLine(draggedElement.UmlClass.Name);
			//Debug.WriteLine(elementCollection.UmlType.Name);
			#endif
			return draggedclassifier.ConformsTo((elementCollection.UmlType as ICollectionType).ElementType);
		}			
		#endregion

		#region Methods to hook in
		private bool dragInitiated;
		private Rectangle startDragArea;
		private int dragZoneSize = 5; // Make property?

		private bool checkValidArea(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			DataGrid dataGrid = sender as DataGrid;
			if (dataGrid == null)
				return true;
			if (e == null) 
				return false;

			System.Windows.Forms.DataGrid.HitTestInfo hti = dataGrid.HitTest(e.X, e.Y);
			return (hti.Type == DataGrid.HitTestType.Cell ||
							hti.Type == DataGrid.HitTestType.RowHeader);
		}
		///<exception cref="ArgumentNullException">Thrown if <paramref name="e"/> is null.</exception>
		protected void OnMouseDown(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			if (e == null) throw new ArgumentNullException("e");
			if (!EffectiveEnabled) return;
			// Only do left-click drags
			if (e.Button != MouseButtons.Left) 
				return;

			dragInitiated = checkValidArea(sender, e);
			if (dragInitiated)
				startDragArea = new Rectangle(e.X - dragZoneSize, e.Y - dragZoneSize, dragZoneSize * 2, dragZoneSize * 2);
		}
		protected void OnMouseUp(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			if (!EffectiveEnabled)
				return;
			dragInitiated = false;
		}
		protected void OnMouseMove(object sender, System.Windows.Forms.MouseEventArgs e)
		{
			if (!EffectiveEnabled)
				return;

			if (e == null) throw new ArgumentNullException("e");
			if (sender == null) throw new ArgumentNullException("sender");
			if (dragInitiated && !startDragArea.Contains(e.X, e.Y))
			{
				dragInitiated = false;
				Control control = sender as Control;
				IElement draggedElement = CurrencyManagerHandle.CurrentElement(control);
				if (draggedElement != null && control != null)
					control.DoDragDrop(new DraggedElement(draggedElement), DragDropEffects.Link);
			}
		}
		protected void OnDragOver(object sender, System.Windows.Forms.DragEventArgs e)
		{
			if (e == null) throw new ArgumentNullException("e");
			bool validDrop = EffectiveEnabled && EcoWinFormHelper.ContainsDroppable(e);
			if (validDrop)
				validDrop = !EcoWinFormHelper.GetIsReadOnly(sender);
				
			IElementCollection targetList = null;
			if (validDrop)
			{
				targetList = EcoWinFormHelper.GetElementCollectionFromDataSource(sender);
				validDrop = (targetList != null) && !targetList.ReadOnly;
			}

			if (validDrop)
			{
				IElement draggedElement = EcoWinFormHelper.GetDroppedElement(e);
				validDrop = ElementConformsToList(draggedElement, targetList);
			}
			e.Effect = validDrop ? AppropriateEffect(e) : DragDropEffects.None;

			// Get the index of the item the mouse is below.
			// The mouse locations are relative to the screen, so they must be
			// converted to client coordinates.
			//int indexOfItemUnderMouseToDrop =
			//	((ListBox)sender).IndexFromPoint(((Control)sender).PointToClient(new Point(e.X, e.Y)));
		}
		protected void OnDragDrop(object sender, System.Windows.Forms.DragEventArgs e)
		{
			if (!EffectiveEnabled) 
			  return;

			if (e == null) throw new ArgumentNullException("e");
			IElement draggedElement = EcoWinFormHelper.GetDroppedElement(e);
			IElementCollection elementCollection = EcoWinFormHelper.GetElementCollectionFromDataSource(sender);

			if (ElementConformsToList(draggedElement, elementCollection))
				elementCollection.Add(draggedElement);
		}
		#endregion

		#region extender's own properties
		#region Enabled property
		private bool enabled = true;
		///<summary>
		///If Enabled is set to false, the drag and drop mechanisms provided by the extender are not active.
		///</summary>
		[Browsable(true)]
		[DefaultValue(true)]
		[LocalizableCategory(typeof(EcoDragDropExtender), "sCategoryBehaviour")]
		[LocalizableDescription(typeof(EcoDragDropExtender), "sPropertyExtenderEnabled")]
		public bool Enabled
		{
			get { return enabled; }
			set { enabled = value; }
		}
		private bool EffectiveEnabled
		{
			get { return Enabled; }
		}
		#endregion

		///<summary>
		///Obsolete, for backwards compatibility. No longer does anything
		///</summary>
		[Browsable(false)]
		[DesignerSerializationVisibility(DesignerSerializationVisibility.Hidden)]
		public EcoSpace EcoSpace
		{
			set {}
		}

		#endregion
	}
}
